//: ## Error Handling
//:
//: Swift has a Result enum.
//:
//: Result is an enum with a success case and a failure case.
//:
//: -------------------
//:
//: [◀  Previous Page](@previous)  |  [Next Page  ▶](@next)
//:
//: -------------------
//:


// re-declaration of Result enum
enum Result<T, E> {
    case success<T>
    case failure<E>
}


//:
//: -------------------
//:
//: Create an Error enum.
//:
//: -------------------
//:


// error enum
enum APIError: Error {
    case badURL
    case parsingError
    case badRequest
    case unknown
}

struct SocialMediaPost: Codable {
    let userId: Int
    let id: Int
    let title: String
    let body: String
}

class SocialFeedAPI {
    
    func fetchLatestFeedFromRemoteAPI() async -> Result<[SocialMediaPost], APIError> {
        guard let endPointUrl = URL(string: "https://jsonplaceholder.typicode.com/posts") else {
            return .failure(.badURL)
        }
        do {
            let (data, _) = try await URLSession.shared.data(from: endPointUrl)
            let socialMediaFeed = try JSONDecoder().decode([SocialMediaPost].self, from: data)
            return .success(socialMediaFeed)
        } catch {
            return .failure(.parsingError)
        }
    }
}


//:
//: -------------------
//:
//: ## In Action
//:
//: -------------------
//:


// main thread
print("Fetching social media feed...")

// new thread
Task {
    let profileAPI = SocialFeedAPI()
    let result = await profileAPI.fetchLatestFeedFromRemoteAPI()
    print("response Received")
    
    // did we have any errors?
    switch result {
    case .success(let socialMediaFeed):
        print("Success")
        print(socialMediaFeed)
    case .failure(let error):
        print("Failure")
        print(error)
    }
}


//:
//: -------------------
//:
//: [◀  Previous Page](@previous)  |  [Next Page  ▶](@next)
//:
//: -------------------
//:
//: ## ![3 Days Of Swift Logo](3DaysIcon146.png) 3 Days Of Swift
//:
//: © Copyright. All rights reserved.
//:
//: 🧕🏻🙋🏽‍♂️👨🏿‍💼👩🏼‍💼👩🏻‍💻💁🏼‍♀️👨🏼‍💼🙋🏻‍♂️🙋🏻‍♀️👩🏼‍💻🙋🏿💁🏽‍♂️🙋🏽‍♀️🙋🏿‍♀️🧕🏾🙋🏼‍♂️
//:
//: Welcome to our community of [3DaysOfSwift.com](https://www.3DaysOfSwift.com) developers!
